import 'package:flutter/material.dart';
import 'package:flutter_svg/flutter_svg.dart';
import 'package:get/get.dart';
import 'package:viserpay_agent/core/utils/dimensions.dart';
import 'package:viserpay_agent/core/utils/my_color.dart';
import 'package:viserpay_agent/core/utils/my_images.dart';
import 'package:viserpay_agent/core/utils/my_strings.dart';
import 'package:viserpay_agent/core/utils/style.dart';
import 'package:viserpay_agent/data/controller/auth/auth/sms_verification_controler.dart';
import 'package:viserpay_agent/data/repo/auth/sms_email_verification_repo.dart';
import 'package:viserpay_agent/data/services/api_service.dart';
import 'package:viserpay_agent/view/components/app-bar/custom_appbar.dart';
import 'package:viserpay_agent/view/components/buttons/gradient_rounded_button.dart';
import 'package:viserpay_agent/view/components/text/default_text.dart';

import '../../../components/otp_field_widget/otp_field_widget.dart';

class SmsVerificationScreen extends StatefulWidget {
  const SmsVerificationScreen({super.key});

  @override
  State<SmsVerificationScreen> createState() => _SmsVerificationScreenState();
}

class _SmsVerificationScreenState extends State<SmsVerificationScreen> {
  @override
  void initState() {
    Get.put(ApiClient(sharedPreferences: Get.find()));
    Get.put(SmsEmailVerificationRepo(apiClient: Get.find()));
    final controller = Get.put(SmsVerificationController(repo: Get.find()));
    super.initState();
    WidgetsBinding.instance.addPostFrameCallback((_) {
      controller.isProfileCompleteEnable = Get.arguments[0];
      controller.loadBefore();
    });
  }

  @override
  void dispose() {
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    return SafeArea(
      child: Scaffold(
        backgroundColor: MyColor.getScreenBgColor(isWhite: true),
        appBar: CustomAppBar(
          fromAuth: true,
          title: MyStrings.smsVerification.tr,
          isShowBackBtn: true,
          bgColor: MyColor.getAppBarColor(),
        ),
        body: GetBuilder<SmsVerificationController>(
          builder: (controller) => controller.isLoading
              ? Center(child: CircularProgressIndicator(color: MyColor.getPrimaryColor()))
              : SingleChildScrollView(
                  padding: Dimensions.screenPaddingHV,
                  child: Center(
                    child: Column(
                      mainAxisAlignment: MainAxisAlignment.center,
                      crossAxisAlignment: CrossAxisAlignment.center,
                      children: [
                        const SizedBox(height: Dimensions.space30),
                        Container(
                          height: 100,
                          width: 100,
                          alignment: Alignment.center,
                          decoration: BoxDecoration(color: MyColor.primaryColor.withOpacity(.075), shape: BoxShape.circle),
                          child: SvgPicture.asset(
                            MyImages.emailVerifyImage,
                            height: 50,
                            width: 50,
                            colorFilter: ColorFilter.mode(MyColor.getPrimaryColor(), BlendMode.srcIn),
                          ),
                        ),
                        const SizedBox(height: Dimensions.space50),
                        Padding(
                          padding: EdgeInsets.symmetric(horizontal: MediaQuery.of(context).size.width * .07),
                          child: DefaultText(text: MyStrings.smsVerificationMsg.tr, maxLines: 3, textAlign: TextAlign.center, textStyle: regularDefault.copyWith(color: MyColor.getLabelTextColor())),
                        ),
                        const SizedBox(height: Dimensions.space10),
                        DefaultText(text: "${MyStrings.phonenumber.tr}: ${controller.maskPhoneNumber(controller.userPhone)}", textAlign: TextAlign.center, textColor: MyColor.getContentTextColor()),
                        const SizedBox(height: 30),
                        OTPFieldWidget(
                          onChanged: (value) {
                            controller.currentText = value;
                          },
                        ),
                        const SizedBox(height: Dimensions.space30),
                        GradientRoundedButton(
                          showLoadingIcon: controller.submitLoading,
                          text: MyStrings.verify.tr,
                          press: () {
                            controller.verifyYourSms(controller.currentText);
                          },
                        ),
                        const SizedBox(height: Dimensions.space30),
                        Row(
                          mainAxisAlignment: MainAxisAlignment.center,
                          children: [
                            Text(MyStrings.didNotReceiveCode.tr, style: regularDefault.copyWith(color: MyColor.getLabelTextColor())),
                            const SizedBox(width: Dimensions.space10),
                            controller.resendLoading
                                ? Container(
                                    margin: const EdgeInsets.all(5),
                                    height: 20,
                                    width: 20,
                                    child: CircularProgressIndicator(
                                      color: MyColor.getPrimaryColor(),
                                    ))
                                : GestureDetector(
                                    onTap: () {
                                      controller.sendCodeAgain();
                                    },
                                    child: Text(MyStrings.resend.tr, style: regularDefault.copyWith(decoration: TextDecoration.underline, color: MyColor.getPrimaryColor()))),
                          ],
                        )
                      ],
                    ),
                  )),
        ),
      ),
    );
  }
}
